<?php
/**
 * Art8io Plugin Updater
 * Checks art8.io server for plugin updates
 */

if (!defined('ABSPATH')) {
    exit;
}

class Art8io_Updater {
    
    private $plugin_slug;
    private $plugin_file;
    private $version;
    private $update_url;
    private $cache_key;
    private $cache_expiry;
    
    public function __construct() {
        $this->plugin_slug = 'art8io';
        $this->plugin_file = 'art8io/art8io.php';
        $this->version = ART8IO_VERSION;
        $this->update_url = 'https://api.art8.io/api/plugins/wordpress/update-info';
        $this->cache_key = 'art8io_update_info';
        $this->cache_expiry = 12 * HOUR_IN_SECONDS;
        
        add_filter('plugins_api', array($this, 'plugin_info'), 20, 3);
        add_filter('site_transient_update_plugins', array($this, 'check_update'));
        add_action('upgrader_process_complete', array($this, 'clear_cache'), 10, 2);
    }
    
    /**
     * Get update info from art8.io server
     */
    private function get_remote_info() {
        $cached = get_transient($this->cache_key);
        if ($cached !== false) {
            return $cached;
        }
        
        $response = wp_remote_get($this->update_url, array(
            'timeout' => 10,
            'headers' => array('Accept' => 'application/json')
        ));
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);
        
        if (empty($data)) {
            return false;
        }
        
        set_transient($this->cache_key, $data, $this->cache_expiry);
        return $data;
    }
    
    /**
     * Check for updates
     */
    public function check_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        $remote_info = $this->get_remote_info();
        
        if ($remote_info && version_compare($this->version, $remote_info->version, '<')) {
            $res = new stdClass();
            $res->slug = $this->plugin_slug;
            $res->plugin = $this->plugin_file;
            $res->new_version = $remote_info->version;
            $res->tested = $remote_info->tested;
            $res->package = $remote_info->download_url;
            $res->url = 'https://art8.io';
            
            $transient->response[$this->plugin_file] = $res;
        }
        
        return $transient;
    }
    
    /**
     * Plugin info popup
     */
    public function plugin_info($result, $action, $args) {
        if ($action !== 'plugin_information' || $args->slug !== $this->plugin_slug) {
            return $result;
        }
        
        $remote_info = $this->get_remote_info();
        
        if (!$remote_info) {
            return $result;
        }
        
        $info = new stdClass();
        $info->name = $remote_info->name;
        $info->slug = $remote_info->slug;
        $info->version = $remote_info->version;
        $info->tested = $remote_info->tested;
        $info->requires = $remote_info->requires;
        $info->requires_php = $remote_info->requires_php;
        $info->author = '<a href="https://art8.io">art8.io</a>';
        $info->author_profile = 'https://art8.io';
        $info->download_link = $remote_info->download_url;
        $info->trunk = $remote_info->download_url;
        $info->last_updated = $remote_info->last_updated;
        $info->sections = array(
            'description' => $remote_info->sections->description,
            'changelog' => $remote_info->sections->changelog
        );
        
        if (!empty($remote_info->banners)) {
            $info->banners = array(
                'low' => $remote_info->banners->low,
                'high' => $remote_info->banners->high
            );
        }
        
        return $info;
    }
    
    /**
     * Clear cache after update
     */
    public function clear_cache($upgrader, $options) {
        if ($options['action'] === 'update' && $options['type'] === 'plugin') {
            delete_transient($this->cache_key);
        }
    }
}
