<?php
if (!defined('ABSPATH')) exit;

class Art8io_Cron {

    public function __construct() {
        add_filter('cron_schedules', array($this, 'add_cron_intervals'));
        add_action('art8io_sync_cron', array($this, 'sync'));
    }

    public function add_cron_intervals($schedules) {
        $interval = intval(get_option('art8io_sync_interval', 15));
        $schedules['art8io_interval'] = array(
            'interval' => $interval * 60,
            'display' => sprintf('Alle %d Minuten', $interval)
        );
        return $schedules;
    }

    public function sync() {
        $api = new Art8io_API();
        $result = $api->get_queue('pending', 20);

        if (is_wp_error($result)) {
            $this->log('sync', $result->get_error_message(), 'error');
            return false;
        }

        $items = isset($result['items']) ? $result['items'] : array();

        if (empty($items)) {
            $this->log('sync', 'Keine neuen Inhalte', 'info');
            update_option('art8io_last_sync', current_time('mysql'));
            return true;
        }

        $auto_draft = false; // Auto-import disabled
        $created = 0;

        foreach ($items as $item) {
            if ($auto_draft) {
                $post_id = $this->create_draft($item);
                if ($post_id) {
                    $api->mark_pushed($item['id']);
                    $created++;
                }
            }
        }

        $this->log('sync', sprintf('%d Drafts erstellt', $created), 'success');
        update_option('art8io_last_sync', current_time('mysql'));

        return true;
    }

    public function create_draft($item, $category_id = 0) {
        // Get import settings FIRST
        $settings = isset($item['import_settings']) ? $item['import_settings'] : array();
        
        // Parse JSON content if needed
        $content_data = $item['content'];
        $parsed_content = '';
        $parsed_title = '';
        $parsed_meta = array();
        
        if (is_string($content_data)) {
            $parsed = json_decode($content_data, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($parsed)) {
                // Extract actual content
                $parsed_content = isset($parsed['content']) ? $parsed['content'] : '';
                
                // Use title from JSON (the real title), not from item (the task)
                $parsed_title = isset($parsed['title']) ? $parsed['title'] : '';
                
                // Extract metadata
                $parsed_meta = array(
                    'meta_title' => isset($parsed['meta_title']) ? $parsed['meta_title'] : '',
                    'meta_description' => isset($parsed['meta_description']) ? $parsed['meta_description'] : '',
                    'focus_keyword' => isset($parsed['focus_keyword']) ? $parsed['focus_keyword'] : '',
                );
            } else {
                $parsed_content = $content_data;
                $parsed_title = isset($item['title']) ? $item['title'] : '';
            }
        } else {
            $parsed_content = $content_data;
            $parsed_title = isset($item['title']) ? $item['title'] : '';
        }
        
        // Fallback to item title if no parsed title
        if (empty($parsed_title)) {
            $parsed_title = isset($item['title']) ? $item['title'] : 'Ohne Titel';
        }
        
        // Convert markdown to HTML
        $parsed_content = $this->convert_markdown($parsed_content, $parsed_title);

        // Get post type from settings - THIS IS THE FIX
        $post_type = 'post';
        if (!empty($settings['post_type'])) {
            $post_type = $settings['post_type'];
        } else {
            $post_type = get_option('art8io_post_type', 'post');
        }
        
        $post_status = !empty($settings['post_status']) ? $settings['post_status'] : 'draft';
        $author_id = !empty($settings['author_id']) ? intval($settings['author_id']) : get_current_user_id();
        $default_category = get_option('art8io_default_category', 0);

        // Get metadata
        $metadata = !empty($item['metadata']) ? $item['metadata'] : $parsed_meta;
        $meta_title = !empty($settings['meta_title']) ? $settings['meta_title'] : (!empty($metadata['meta_title']) ? $metadata['meta_title'] : '');
        $meta_desc = !empty($settings['meta_description']) ? $settings['meta_description'] : (!empty($metadata['meta_description']) ? $metadata['meta_description'] : '');
        $focus_kw = !empty($settings['focus_keyword']) ? $settings['focus_keyword'] : (!empty($metadata['focus_keyword']) ? $metadata['focus_keyword'] : '');

        // Prepare post data
        $post_data = array(
            'post_title' => sanitize_text_field($parsed_title),
            'post_content' => wp_kses_post($parsed_content),
            'post_status' => $post_status,
            'post_type' => $post_type,
            'post_author' => $author_id,
            'meta_input' => array(
                '_art8io_id' => $item['id'],
                '_art8io_imported' => current_time('mysql'),
            ),
        );

        // Category for posts
        $cat_id = $category_id ?: intval($settings['category_id'] ?? 0) ?: $default_category;
        if ($cat_id && $post_type === 'post') {
            $post_data['post_category'] = array($cat_id);
        }

        // Product category for WooCommerce
        if ($post_type === 'product' && $cat_id) {
            $post_data['tax_input'] = array('product_cat' => array($cat_id));
        }

        // Page parent - THIS IS THE FIX FOR PAGES
        if ($post_type === 'page' && !empty($settings['parent_id'])) {
            $post_data['post_parent'] = intval($settings['parent_id']);
        }

        // Tags
        if (!empty($metadata['tags']) && is_array($metadata['tags'])) {
            $post_data['tags_input'] = $metadata['tags'];
        }

        // Debug log
        $this->log('create_draft', sprintf('Creating %s: "%s" (post_type from settings: %s)', $post_type, $parsed_title, $settings['post_type'] ?? 'not set'), 'info');

        // Create post
        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            $this->log('create_draft', $post_id->get_error_message(), 'error');
            return false;
        }

        // SEO Meta
        if ($meta_title || $meta_desc) {
            $this->set_seo_meta($post_id, $meta_title, $meta_desc);
        }

        // Focus Keyword
        if ($focus_kw) {
            $this->set_focus_keyword($post_id, $focus_kw);
        }

        $this->log('create_draft', sprintf('"%s" erstellt (ID: %d, Typ: %s)', $parsed_title, $post_id, $post_type), 'success');
        return $post_id;
    }

    private function convert_markdown($text, $title = '') {
        if (empty($text)) return $text;
        
        // Remove first heading if it matches title
        if (!empty($title)) {
            $text = preg_replace('/^#+ ?' . preg_quote($title, '/') . '\s*/i', '', trim($text));
        }
        
        // Headers
        $text = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $text);
        $text = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $text);
        $text = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $text);
        
        // Bold
        $text = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $text);
        
        // Italic
        $text = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $text);
        
        // Lists
        $text = preg_replace('/^- (.+)$/m', '<li>$1</li>', $text);
        $text = preg_replace('/(<li>.*<\/li>\n?)+/', '<ul>$0</ul>', $text);
        
        // Line breaks
        $text = str_replace('\n\n', '</p><p>', $text);
        $text = str_replace('\n', '<br>', $text);
        $text = preg_replace('/\n\n+/', '</p><p>', $text);
        $text = str_replace("\n", '<br>', $text);
        
        return trim($text);
    }

    private function set_seo_meta($post_id, $meta_title, $meta_description) {
        if (defined('WPSEO_VERSION')) {
            if ($meta_title) update_post_meta($post_id, '_yoast_wpseo_title', $meta_title);
            if ($meta_description) update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
            return;
        }
        if (defined('RANK_MATH_VERSION')) {
            if ($meta_title) update_post_meta($post_id, 'rank_math_title', $meta_title);
            if ($meta_description) update_post_meta($post_id, 'rank_math_description', $meta_description);
            return;
        }
        if (defined('AIOSEO_VERSION')) {
            if ($meta_title) update_post_meta($post_id, '_aioseo_title', $meta_title);
            if ($meta_description) update_post_meta($post_id, '_aioseo_description', $meta_description);
            return;
        }
    }

    private function set_focus_keyword($post_id, $keyword) {
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $keyword);
        } elseif (defined('RANK_MATH_VERSION')) {
            update_post_meta($post_id, 'rank_math_focus_keyword', $keyword);
        }
    }

    private function log($action, $message, $status) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'art8io_logs';
        $wpdb->insert($table_name, array(
            'action' => $action,
            'message' => $message,
            'status' => $status,
        ), array('%s', '%s', '%s'));
    }

    public static function trigger_sync() {
        $instance = new self();
        return $instance->sync();
    }
}
