<?php
/**
 * Art8io API Client
 */

if (!defined('ABSPATH')) {
    exit;
}

class Art8io_API {
    
    private $api_url;
    private $api_key;
    
    public function __construct() {
        $this->api_url = ART8IO_API_URL;
        $this->api_key = get_option('art8io_api_key', '');
    }
    
    private function request($endpoint, $method = 'GET', $body = null) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', 'Kein API Key konfiguriert');
        }
        
        $args = array(
            'method' => $method,
            'timeout' => 30,
            'headers' => array(
                'X-API-Key' => $this->api_key,
                'Content-Type' => 'application/json',
            ),
        );
        
        if ($body && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = json_encode($body);
        }
        
        $response = wp_remote_request($this->api_url . $endpoint, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status_code >= 400) {
            $message = isset($data['detail']) ? $data['detail'] : 'API Fehler';
            return new WP_Error('api_error', $message, array('status' => $status_code));
        }
        
        return $data;
    }
    
    public function validate_key($api_key = null) {
        if ($api_key) {
            $this->api_key = $api_key;
        }
        if (empty($this->api_key)) {
            return false;
        }
        $result = $this->request('/queue?limit=1');
        return !is_wp_error($result);
    }
    
    public function get_queue($status = 'pending', $limit = 10) {
        return $this->request('/queue?status=' . urlencode($status) . '&limit=' . intval($limit));
    }
    
    public function mark_pushed($item_id) {
        return $this->request('/queue/' . $item_id . '/mark-pushed', 'POST');
    }
    
    public function delete_item($item_id) {
        return $this->request('/queue/' . $item_id, 'DELETE');
    }

    public function get_status() {
        $is_connected = false;
        $api_key_set = !empty($this->api_key);
        
        if ($api_key_set) {
            $is_connected = $this->validate_key();
        }
        
        return array(
            'connected' => $is_connected,
            'api_key_set' => $api_key_set,
            'last_sync' => get_option('art8io_last_sync', ''),
        );
    }
    
    /**
     * Deactivate the API key on art8.io server
     */
    public function deactivate_key() {
        if (empty($this->api_key)) {
            return false;
        }
        
        $result = $this->request('/keys/deactivate', 'POST');
        return !is_wp_error($result);
    }
}
