<?php
if (!defined('ABSPATH')) exit;

class Art8io_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_art8io_sync_now', array($this, 'ajax_sync_now'));
        add_action('wp_ajax_art8io_validate_key', array($this, 'ajax_validate_key'));
        add_action('wp_ajax_art8io_import_single', array($this, 'ajax_import_single'));
        add_action('wp_ajax_art8io_delete_item', array($this, 'ajax_delete_item'));
    }
    
    public function add_menu() {
        // Get queue count for notification bubble
        $queue_count = 0;
        $api_key = get_option("art8io_api_key");
        if ($api_key) {
            $api = new Art8io_API();
            $queue = $api->get_queue("pending", 100);
            if ($queue && !is_wp_error($queue) && isset($queue["items"])) {
                $queue_count = count($queue["items"]);
            }
        }
        $bubble = $queue_count > 0 ? " <span class=\"awaiting-mod\">" . $queue_count . "</span>" : "";
        $queue_bubble = $queue_count > 0 ? " <span class=\"awaiting-mod\">" . $queue_count . "</span>" : "";
        
        add_menu_page("art8.io", "art8.io" . $bubble, "manage_options", "art8io", array($this, "render_dashboard"), "dashicons-cloud-upload", 30);
        add_submenu_page("art8io", "Dashboard", "Dashboard", "manage_options", "art8io", array($this, "render_dashboard"));
        add_submenu_page("art8io", "Content Queue", "Content Queue" . $queue_bubble, "manage_options", "art8io-queue", array($this, "render_queue"));
        add_submenu_page("art8io", "Einstellungen", "Einstellungen", "manage_options", "art8io-settings", array($this, "render_settings"));
        add_submenu_page("art8io", "Logs", "Logs", "manage_options", "art8io-logs", array($this, "render_logs"));
    }
    
    public function register_settings() {
        register_setting('art8io_settings', 'art8io_api_key');
        register_setting('art8io_settings', 'art8io_sync_interval');
        register_setting('art8io_settings', 'art8io_post_type');
        register_setting('art8io_settings', 'art8io_default_category');
    }
    
    public function enqueue_assets($hook) {
        if (strpos($hook, 'art8io') === false) return;
        wp_enqueue_style('art8io-admin', ART8IO_PLUGIN_URL . 'assets/css/admin.css', array(), ART8IO_VERSION);
        wp_enqueue_script('art8io-admin', ART8IO_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), ART8IO_VERSION, true);
        wp_localize_script('art8io-admin', 'art8io', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('art8io_nonce'),
            'strings' => array(
                'syncing' => 'Synchronisiere...',
                'validating' => 'Prüfe...',
                'valid' => 'Gültig!',
                'invalid' => 'Ungültig',
            )
        ));
    }

    public function render_dashboard() {
        $api = new Art8io_API();
        $status = $api->get_status();
        $pending = 0;
        if ($status['connected']) {
            $q = $api->get_queue('pending', 100);
            if ($q && !is_wp_error($q) && isset($q['items'])) $pending = count($q['items']);
        }
        $drafts = count(get_posts(array('post_type' => get_option('art8io_post_type', 'post'), 'post_status' => 'draft', 'meta_key' => '_art8io_id', 'posts_per_page' => -1)));
        ?>
        <div class="wrap art8io-wrap">
            <h1 class="art8io-title">art8.io</h1>
            
            <div class="art8io-card <?php echo $status['connected'] ? 'art8io-connected' : ''; ?>">
                <div class="art8io-card-header">
                    <div class="art8io-status-row">
                        <span class="art8io-dot <?php echo $status['connected'] ? 'active' : ''; ?>"></span>
                        <div>
                            <strong><?php echo $status['connected'] ? 'Verbunden' : 'Nicht verbunden'; ?></strong>
                            <?php if ($status['last_sync']): ?>
                                <br><small>Letzte Sync: <?php echo date_i18n('d.m.Y H:i', strtotime($status['last_sync'])); ?></small>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php if ($status['connected']): ?>
                        <a href="https://art8.io/dashboard/plugins" target="_blank" class="button">Verwalten</a>
                    <?php endif; ?>
                </div>
                <?php if (!$status["connected"]): ?>
                <div class="art8io-onboarding" style="padding:20px 0;">
                    <h2 style="margin-top:0;">🚀 Willkommen bei art8.io!</h2>
                    <p>Verbinde dein WordPress mit art8.io um automatisch Content zu empfangen.</p>
                    <h3>So gehts:</h3>
                    <ol style="line-height:2;">
                        <li><strong>API Key erstellen:</strong> Gehe zu <a href="https://art8.io/dashboard/plugins" target="_blank">art8.io Dashboard → Plugins</a> und erstelle einen neuen API Key für WordPress.</li>
                        <li><strong>Key einfügen:</strong> Kopiere den Key und füge ihn in den <a href="<?php echo admin_url("admin.php?page=art8io-settings"); ?>">Einstellungen</a> ein.</li>
                        <li><strong>Fertig!</strong> Dein Content wird automatisch synchronisiert.</li>
                    </ol>
                    <div style="margin-top:20px;">
                        <a href="<?php echo admin_url("admin.php?page=art8io-settings"); ?>" class="button button-primary button-hero">Jetzt verbinden</a>
                        <a href="https://art8.io/dashboard/plugins" target="_blank" class="button button-hero" style="margin-left:10px;">API Key erstellen</a>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <?php if ($status['connected']): ?>
            <div class="art8io-stats">
                <div class="art8io-stat"><span class="num"><?php echo $pending; ?></span><span class="label">In Queue</span></div>
                <div class="art8io-stat"><span class="num"><?php echo $drafts; ?></span><span class="label">Drafts</span></div>
            </div>
            <div class="art8io-card">
                <button type="button" class="button button-primary art8io-sync-btn">
                    <span class="dashicons dashicons-update"></span> Jetzt synchronisieren
                </button>
                <a href="<?php echo admin_url('admin.php?page=art8io-queue'); ?>" class="button">Queue anzeigen</a>
                <div class="art8io-sync-result"></div>
            </div>
            <div class="art8io-card art8io-version-card">
                <div style="display:flex;justify-content:space-between;align-items:center;">
                    <div><span style="color:#666;font-size:12px;">Version</span><br><strong><?php echo ART8IO_VERSION; ?></strong></div>
                    <div style="text-align:right;font-size:12px;color:#666;"><a href="https://art8.io" target="_blank">art8.io</a> · <a href="<?php echo admin_url('update-core.php'); ?>">Updates</a></div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    public function render_queue() {
        include ART8IO_PLUGIN_DIR . 'admin/views/queue.php';
    }

    public function render_settings() {
        $api = new Art8io_API();
        $status = $api->get_status();
        $seo = 'none';
        if (defined('WPSEO_VERSION')) $seo = 'Yoast SEO';
        elseif (defined('RANK_MATH_VERSION')) $seo = 'RankMath';
        elseif (defined('AIOSEO_VERSION')) $seo = 'All in One SEO';
        ?>
        <div class="wrap art8io-wrap">
            <h1>Einstellungen</h1>
            <form method="post" action="options.php">
                <?php settings_fields('art8io_settings'); ?>
                
                <div class="art8io-card">
                    <h3>Verbindung</h3>
                    <table class="form-table">
                        <tr>
                            <th>API Key</th>
                            <td>
                                <div class="art8io-key-row">
                                    <input type="password" name="art8io_api_key" id="art8io_api_key" value="<?php echo esc_attr(get_option('art8io_api_key')); ?>" class="regular-text" placeholder="art8_...">
                                    <button type="button" class="button art8io-toggle-key"><span class="dashicons dashicons-visibility"></span></button>
                                    <button type="button" class="button art8io-validate-key-btn">Testen</button>
                                </div>
                                <span class="art8io-key-status <?php echo $status['connected'] ? 'valid' : ''; ?>">
                                    <?php if ($status['connected']): ?><span class="dashicons dashicons-yes-alt"></span> Verbunden<?php endif; ?>
                                </span>
                                <p class="description">Aus <a href="https://art8.io/dashboard/plugins" target="_blank">art8.io Dashboard</a></p>
                            </td>
                        </tr>
                    </table>
                    <?php if ($status['connected']): ?>
                        <hr>
                        <p class="description">Verbindung verwalten im <a href="https://art8.io/dashboard/plugins" target="_blank">art8.io Dashboard</a></p>
                    <?php endif; ?>
                </div>

                <div class="art8io-card">
                    <h3>Standard Import-Einstellungen</h3>
                    <table class="form-table">
                        <tr>
                            <th>Post-Typ</th>
                            <td>
                                <select name="art8io_post_type">
                                    <?php foreach (get_post_types(array('public' => true), 'objects') as $pt): if ($pt->name === 'attachment') continue; ?>
                                        <option value="<?php echo $pt->name; ?>" <?php selected(get_option('art8io_post_type', 'post'), $pt->name); ?>><?php echo $pt->label; ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">Kann beim Import in der Queue überschrieben werden.</p>
                            </td>
                        </tr>
                        <tr>
                            <th>Standard-Kategorie</th>
                            <td><?php wp_dropdown_categories(array('name' => 'art8io_default_category', 'selected' => get_option('art8io_default_category', 0), 'show_option_none' => 'Keine', 'option_none_value' => 0, 'hide_empty' => false)); ?></td>
                        </tr>
                        <tr>
                        <tr>
                            <th>Sync-Intervall</th>
                            <td>
                                <select name="art8io_sync_interval">
                                    <option value="15" <?php selected(get_option('art8io_sync_interval', '15'), '15'); ?>>15 Minuten</option>
                                    <option value="30" <?php selected(get_option('art8io_sync_interval'), '30'); ?>>30 Minuten</option>
                                    <option value="60" <?php selected(get_option('art8io_sync_interval'), '60'); ?>>Stündlich</option>
                                </select>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="art8io-card">
                    <h3>SEO Integration</h3>
                    <p>Meta Title, Description und Focus Keyword werden automatisch übertragen.</p>
                    <p><strong>Erkanntes Plugin:</strong> 
                        <?php if ($seo !== 'none'): ?>
                            <span class="art8io-badge green"><span class="dashicons dashicons-yes"></span> <?php echo $seo; ?></span>
                        <?php else: ?>
                            <span class="art8io-badge" style="background:#fcf0f1;color:#d63638;"><span class="dashicons dashicons-warning"></span> Kein SEO Plugin</span>
                            <p class="description" style="color:#d63638;">Installiere Yoast SEO, RankMath oder All in One SEO für die automatische Übernahme von Meta-Daten.</p>
                        <?php endif; ?>
                    </p>
                </div>
                <div class="art8io-card">
                    <h3>Shop-Erkennung</h3>
                    <p>WooCommerce erkannt - Content für bessere KI-Sichtbarkeit deines Shops.</p>
                    <p><strong>Erkanntes Plugin:</strong>
                        <?php if (class_exists("WooCommerce")): ?>
                            <span class="art8io-badge green"><span class="dashicons dashicons-yes"></span> WooCommerce <?php echo defined("WC_VERSION") ? WC_VERSION : ""; ?></span>
                        <?php else: ?>
                            <span class="art8io-badge" style="background:#f0f0f1;color:#50575e;"><span class="dashicons dashicons-info"></span> Kein Shop Plugin</span>
                            <p class="description">Optional: Installiere WooCommerce um Produkte zu importieren.</p>
                        <?php endif; ?>
                    </p>
                </div>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function render_logs() {
        global $wpdb;
        $logs = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}art8io_logs ORDER BY created_at DESC LIMIT 100");
        ?>
        <div class="wrap art8io-wrap">
            <h1>Logs</h1>
            <?php if (empty($logs)): ?>
                <div class="art8io-empty"><span class="dashicons dashicons-media-text"></span><h2>Keine Logs</h2></div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead><tr><th style="width:150px">Datum</th><th style="width:100px">Aktion</th><th>Nachricht</th><th style="width:80px">Status</th></tr></thead>
                    <tbody>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td><?php echo date_i18n('d.m.Y H:i', strtotime($log->created_at)); ?></td>
                                <td><?php echo esc_html($log->action); ?></td>
                                <td><?php echo esc_html($log->message); ?></td>
                                <td><span class="art8io-badge <?php echo $log->status === 'success' ? 'green' : ($log->status === 'error' ? 'red' : 'gray'); ?>"><?php echo $log->status; ?></span></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
    }

    // AJAX: Sync now
    public function ajax_sync_now() {
        check_ajax_referer('art8io_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Keine Berechtigung');
        $result = Art8io_Cron::trigger_sync();
        $result ? wp_send_json_success('Synchronisation erfolgreich') : wp_send_json_error('Fehlgeschlagen');
    }

    // AJAX: Validate key
    public function ajax_validate_key() {
        check_ajax_referer('art8io_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Keine Berechtigung');
        $key = sanitize_text_field($_POST['api_key'] ?? '');
        if (!$key) wp_send_json_error('Kein Key');
        $api = new Art8io_API();
        $api->validate_key($key) ? wp_send_json_success('Gültig!') : wp_send_json_error('Ungültig');
    }

    // AJAX: Import single with advanced settings
    public function ajax_import_single() {
        check_ajax_referer('art8io_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Keine Berechtigung');
        
        $item = json_decode(stripslashes($_POST['item'] ?? ''), true);
        $settings = json_decode(stripslashes($_POST['settings'] ?? '{}'), true);
        
        if (!$item) wp_send_json_error('Kein Content');
        
        // Merge settings into item for processing
        if (!empty($settings)) {
            $item['import_settings'] = $settings;
        }
        
        $cron = new Art8io_Cron();
        $post_id = $cron->create_draft($item, intval($settings['category_id'] ?? 0));
        
        if ($post_id) {
            $api = new Art8io_API();
            $api->mark_pushed($item['id']);
            wp_send_json_success(array(
                'message' => 'Importiert!',
                'post_id' => $post_id,
                'edit_url' => get_edit_post_link($post_id, 'raw')
            ));
        }
        wp_send_json_error('Import fehlgeschlagen');
    }

    // AJAX: Delete item
    public function ajax_delete_item() {
        check_ajax_referer('art8io_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Keine Berechtigung');
        $item_id = sanitize_text_field($_POST['item_id'] ?? '');
        if (!$item_id) wp_send_json_error('Keine ID');
        $api = new Art8io_API();
        $result = $api->delete_item($item_id);
        $result ? wp_send_json_success('Gelöscht') : wp_send_json_error('Fehler beim Löschen');
    }
}
